// background.js

function successSound() {
  const audioContext = new (window.AudioContext || window.webkitAudioContext)();
  const audioFileURL = chrome.runtime.getURL('sound.mp3');
  const request = new XMLHttpRequest();
  request.open('GET', audioFileURL, true);
  request.responseType = 'arraybuffer';
  request.onload = function () {
    audioContext.decodeAudioData(request.response, function (buffer) {
      const source = audioContext.createBufferSource();
      source.buffer = buffer;
      source.connect(audioContext.destination);
      source.start(0);
    });
  };
  request.send();
}

function saveDataToStorage(data) {
  return new Promise((resolve, reject) => {
    chrome.storage.local.set(data, () => {
      if (chrome.runtime.lastError) {
        reject(chrome.runtime.lastError);
      } else {
        resolve();
      }
    });
  });
}

function getDataFromStorage(keys) {
  return new Promise((resolve, reject) => {
    chrome.storage.local.get(keys, (result) => {
      if (chrome.runtime.lastError) {
        reject(chrome.runtime.lastError);
      } else {
        resolve(result);
      }
    });
  });
}

// ADILET FUNCTIONS
async function adilet_processItem(item, page = 0) {
  return new Promise(async (resolve, reject) => {
    const url = 'https://aisoip.adilet.gov.kz/extperson/api/rest/execproc/search?page=' + page + "&size=100";
    const body = { iin: item.iin, searchType: false };
    const response = await sendRequest(url, body, "POST");
    console.log(response);
    if (response.status == 200) {
      const result = await response.json();
      console.log(result);
      if (!result.pagination.empty) {
        await adilet_fetchAndDownloadData(item.iin, page + 1, result.pagination.searchId, result.pagination.totalPages);
      }
      if (!result.pagination.last) {
        resolve(await adilet_processItem(item, page + 1));
      } else {
        item.completed = true;
        resolve(item);
      }
    }
  });
}

async function adilet_fetchAndDownloadData(iin, pageNumber, searchId, totalPages) {
  for (let i = 0; i < totalPages; i++) {
    const queryParameters = 'searchtype=false&searchid=' + searchId + "&lang=ru" + "&page=" + i + '&size=1000';
    const downloadOptions = {
      url: 'https://aisoip.adilet.gov.kz/extperson/api/rest/export/excel?' + queryParameters,
      filename: await adilet_generateFilePath(iin, pageNumber, i)
    };
    await new Promise((resolve, reject) => {
      chrome.downloads.download(downloadOptions, function (downloadId) {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          resolve(downloadId);
        }
      });
    });
  }
}

async function adilet_generateFilePath(iin, pageNumber, innerPageNumber) {
  const lastImported = getDataFromStorage(['adilet_lastImportedFile']);
  const lastImportedFile = await lastImported;
  const parentFolder = "Должники-из-файла-" + lastImportedFile;
  const filename = "Должник-" + iin + '-страница-' + pageNumber + ".xlsx";
  const subFolder = "Должник-" + iin;
  return parentFolder + '/' + subFolder + '/' + innerPageNumber + filename;
}

async function sendRequest(url, body, method) {
  try {
    const response = await fetch(url, {
      method,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(body)
    });
    return response;
  } catch (error) {
    console.log(error);
    throw error;
  }
}

async function checkConnection() {
  const url = `https://aisoip.adilet.gov.kz/extperson/api/rest/execproc/search?page=0&size=100`;
  const body = { iin: '', searchType: false };
  const response = await sendRequest(url, body, 'POST');
  try {
    return response.status == 200;
  } catch (error) {
    return false;
  }
}

// STOP FUNCTIONS
const BASE_URL = 'https://aisoip.adilet.gov.kz';

function stop_generateFilePath(itemNum) {
  const normalizedItemNum = itemNum.replace(/\//g, '_');
  const parentFolder = `Результаты-парсера`;
  const filename = `${normalizedItemNum}.pdf`;
  return `${parentFolder}/${filename}`;
}

async function stop_fetchExecProcId(item, page = 0) {
  const url = `${BASE_URL}/extperson/api/rest/execproc/search?page=${page}&size=100`;
  const body = { iin: item.number, searchType: false };
  const response = await sendRequest(url, body, 'POST');
  if (response.status !== 200) {
    throw new Error('Failed to fetch execution procedure ID');
  }
  const result = await response.json();
  console.log(result);
  if (!result.pagination.empty) {
    const matchedItem = result.content.find(obj => obj.execProcNum === item.execProcNum);
    if (matchedItem) {
      return matchedItem.execProcId;
    } else {
      console.log(`No match found for execProcNum: ${item.execProcNum}`);
      return null;
    }
  }
  if (!result.pagination.last) {
    return stop_fetchExecProcId(item, page + 1);
  }
  return null;
}

async function stop_fetchAndDownloadDocument(execProcId, itemNum) {
  const url = `${BASE_URL}/extperson/api/rest/execproc/doc/${execProcId}?searchType=false`;
  const response = await fetch(url);
  if (response.status !== 200) {
    throw new Error('Failed to fetch document details');
  }
  const documents = await response.json();
  const targetDoc = documents.find(doc => doc.ddocTitle.includes("Постановление о прекращении исполнительного производства"));
  if (targetDoc) {
    setTimeout(async () => { 
      const downloadUrl = `${BASE_URL}/extperson/api/rest/execproc/doc/${targetDoc.did}/${targetDoc.ddocName}?lang=ru`;
      stop_downloadDocument(downloadUrl, itemNum);
    }, 30);
  }
}

async function stop_processItem(item) {
  try {
    const execProcId = await stop_fetchExecProcId(item);
    console.log(item.number);
    console.log(execProcId);
    if (execProcId) {
      await stop_fetchAndDownloadDocument(execProcId, item.number);
    }
    return true;
  } catch (error) {
    console.error('Error processing item:', error);
    return false;
  }
}

async function stop_downloadDocument(url, itemNum) {
  const options = { url: url, filename: stop_generateFilePath(itemNum) };
  chrome.downloads.download(options, function(downloadId) {
    if (chrome.runtime.lastError) {
      console.error('Download failed:', chrome.runtime.lastError);
    } else {
      console.log('Download initiated, ID:', downloadId);
    }
  });
}

// CSI FUNCTIONS
async function csi_fetchExecProcId(item, page = 0) {
  const url = `${BASE_URL}/extperson/api/rest/execproc/search?page=${page}&size=5`;
  const body = { iin: item.number, searchType: false };
  const response = await sendRequest(url, body, 'POST');
  if (response.status !== 200) {
    throw new Error('Failed to fetch execution procedure ID');
  }
  const result = await response.json();
  if (!result.pagination.empty) {
    const matchedItem = result.content.find(obj => obj.execProcNum === item.execProcNum);
    if (matchedItem) {
      return matchedItem.execProcId;
    } else {
      console.log(`No match found for execProcNum: ${item.execProcNum}`);
      return null;
    }
  }
  if (!result.pagination.last) {
    return csi_fetchExecProcId(item, page + 1);
  }
  return null;
}

async function csi_fetchAndDownloadDocuments(execProcId, iin) {
  const excelDocTypes = ["request", "doc", "restriction"];
  const downloadedExcelFiles = [];
  for (let docType of excelDocTypes) {
    const downloadUrl = `${BASE_URL}/extperson/api/rest/execproc/excel/${execProcId}?docType=${docType}&lang=ru`;
    const fileResponse = await fetch(downloadUrl);
    if (!fileResponse.ok) {
      throw new Error(`Failed to fetch document: ${docType}`);
    }
    const blob = await fileResponse.blob();
    const arrayBuffer = await blob.arrayBuffer();
    const uint8Array = new Uint8Array(arrayBuffer);
    downloadedExcelFiles.push(uint8Array);
  }
  if (downloadedExcelFiles.length !== 3) {
    throw new Error("Failed to download all three Excel files");
  }
  return downloadedExcelFiles;
}

async function csi_processItem(item) {
  try {
    const execProcId = await csi_fetchExecProcId(item);
    let downloadedExcelFiles = [];
    if (execProcId) {
      downloadedExcelFiles = await csi_fetchAndDownloadDocuments(execProcId, item.execProcNum);
    }
    return [true, downloadedExcelFiles];
  } catch (error) {
    console.error('Error processing item:', error);
    return [false, []];
  }
}

async function csi_downloadDocument(url, itemNum) {
  const options = { url: url, filename: csi_generateFilePath(itemNum) };
  chrome.downloads.download(options, function(downloadId) {
    if (chrome.runtime.lastError) {
      console.error('Download failed:', chrome.runtime.lastError);
    } else {
      console.log('Download initiated, ID:', downloadId);
    }
  });
}

function csi_generateFilePath(itemNum) {
  const normalizedItemNum = itemNum.replace(/\//g, '_');
  const parentFolder = `Результаты-парсера`;
  const filename = `${normalizedItemNum}.pdf`;
  return `${parentFolder}/${filename}`;
}

// SUD FUNCTIONS
function updateUrlAndReloadTab(tabId, newUrl) {
  chrome.tabs.update(tabId, { url: newUrl }, function() {
    chrome.tabs.reload(tabId, function() {
      currentState = 'start';
    });
  });
}

let currentState = 'start';
let curData = null;
let res = null;
let workingTab = null;
let isUserLogged = false;
let responseCallback = null;

async function checkUserLoginInTab(tabId) {
  await chrome.scripting.executeScript({
    target: { tabId: tabId },
    files: ['isLogged.js']
  }, (injectionResults) => {
    if (chrome.runtime.lastError || !injectionResults || !injectionResults.length) {
      console.error('Failed to execute script: ' + chrome.runtime.lastError.message);
      return;
    }
    return isUserLogged;
  });
}

const responseEventTarget = new EventTarget();

function waitForResponse() {
  return new Promise((resolve) => {
    const handler = (event) => {
      console.log("Background received custom event:", event);
      resolve(event.detail);
      responseEventTarget.removeEventListener('responseReceived', handler);
    };
    responseEventTarget.addEventListener('responseReceived', handler);
  });
}

chrome.runtime.onMessage.addListener(function(request, sender, sendResponse) {
  console.log("Background received message:", request);
  if (request.action === 'checkConnection') {
    async function checkConnect() {
      res = await checkConnection();
      console.log(res);
      sendResponse({ found: res, action: 'checkConnectionResponse' });
    }
    checkConnect();
    return true;
  }
  if (request.action === 'getData') {
    async function getData() {
      res = await getDataFromStorage(request.data);
      console.log({ found: true, res: res, action: 'getDataResponse', sudData: curData });
      sendResponse({ found: true, res: res, action: 'getDataResponse', sudData: curData });
    }
    getData();
    return true;
  }
  if (request.action === 'getDataS') {
    async function getData() {
      res = await getDataFromStorage(request.data);
      sendResponse({ data: curData });
    }
    getData();
    return true;
  }
  if (request.action === 'saveData') {
    async function saveData() {
      res = await saveDataToStorage(request.data);
      console.log(res);
      sendResponse({ found: true, res: res, action: 'saveDataResponse' });
    }
    saveData();
    return true;
  }
  if (request.action === 'stop_processItem') {
    async function process() {
      let item = request.item;
      res = await stop_processItem(item);
      console.log(res);
      sendResponse({ found: res, action: 'stop_processItemResponse' });
    }
    process();
    return true;
  }
  if (request.action === 'adilet_processItem') {
    async function process() {
      let item = request.item;
      res = await adilet_processItem(item);
      console.log(res);
      sendResponse({ found: res, action: 'adilet_processItemResponse' });
    }
    process();
    return true;
  }
  if (request.action === 'csi_processItem') {
    async function process() {
      let item = request.item;
      res = await csi_processItem(item);
      console.log(res);
      sendResponse({ found: res[0], action: 'csi_processItemResponse', downloaded: res[1] });
    }
    process();
    return true;
  }
  if (request.action === 'checkTab') {
    async function checkForTargetTab() {
      chrome.tabs.query({}, async function(tabs) {
        const targetTab = tabs.find(tab => tab.url === "https://office.sud.kz/form/proceedings/myCases.xhtml");
        workingTab = targetTab;
        const officeTab = tabs.find(tab => tab.favIconUrl === "https://office.sud.kz/favicon.ico");
        if (targetTab) {
          await checkUserLoginInTab(targetTab.id);
          setTimeout(function (){
            if (!isUserLogged) {
              sendResponse({ found: false, action: 'checkTabResponse' });
            } else {
              sendResponse({ found: true, tabId: targetTab.id, action: 'checkTabResponse' });
            }
            clearInterval(checkInterval);
          }, 1000);
        } else if (!officeTab || officeTab.status === "complete") {
          sendResponse({ found: false, action: 'checkTabResponse' });
          clearInterval(checkInterval);
        }
      });
    }
    const checkInterval = setInterval(checkForTargetTab, 1000);
    return true;
  }
  if (request.action === 'sud_processItem') {
    const itemNum = request.itemNum;
    console.log(itemNum);
    async function processItem() {
      await checkUserLoginInTab(workingTab.id);
      console.log(isUserLogged);
      if (!isUserLogged) {
        sendResponse({ found: false, action: 'sud_processItemResponse' });
      }
      curData = itemNum;
      console.log(workingTab.id);
      console.log(curData);
      await chrome.scripting.executeScript({
        target: { tabId: workingTab.id },
        files: ['contentScript.js']
      });
      try {
        const result = await waitForResponse();
        console.log("Response received:", result);
        sendResponse({ found: true, result, action: 'sud_processItemResponse' });
      } catch (error) {
        console.error("Error while waiting for response:", error);
        sendResponse({ found: false, action: 'sud_processItemResponse', error: error.message });
      }
    }
    processItem();
    return true;
  }
  if (request.action === 'getState') {
    console.log('getState', currentState);
    sendResponse({ state: currentState });
  } else if (request.action === 'setState') {
    currentState = request.state;
    console.log('setState', currentState);
    sendResponse({ state: currentState });
  } else if (request.action === 'getData') {
    console.log('getData', curData);
    sendResponse({ data: curData });
  } else if (request.action === 'setData') {
    curData = request.data;
    console.log('setData', curData);
    sendResponse({ data: curData });
  } else if (request.action === 'setRes') {
    console.log("setting results", request.res);
    res = request.res;
    sendResponse({ res });
    const event = new CustomEvent('responseReceived', { detail: request.res });
    responseEventTarget.dispatchEvent(event);
    curData = null;
  } else if (request.action === 'getRes') {
    sendResponse({ res });
    res = null;
  } else if (request.action === 'startOver') {
    updateUrlAndReloadTab(workingTab.id, "https://office.sud.kz/form/proceedings/myCases.xhtml");
    sendResponse('done');
  } else if (request.action == 'userLogged') {
    isUserLogged = request.isLogged;
    sendResponse('done');
  } else if (request.action === 'responseReady') {
    if (responseCallback) {
      responseCallback(request.res);
      responseCallback = null;
    }
  } else if (request.type === 'setKazpatentFlag') {
    chrome.storage.local.set({ kazpatent_from_easywork: true }, () => {
      sendResponse({ status: 'ok' });
    });
    return true;
  } else if (request.type === 'getKazpatentFlag') {
    chrome.storage.local.get(['kazpatent_from_easywork'], (result) => {
      const flag = !!result.kazpatent_from_easywork;
      if (flag) {
        chrome.storage.local.remove('kazpatent_from_easywork'); // сбрасываем после использования
      }
      sendResponse({ flag });
    });
    return true;
  }
});
